/**
  Copyright (c) 2010 Freescale Semiconductor
  Freescale Confidential Proprietary
  
  \file  	  FontLibrary.h
  \brief	  This is the Font Library Driver File
  \brief	  Library for rendering text from Freescale Font files
  \brief	  Uses DMA for memory transfers. Supports 8bpp
  \author	  Freescale Semiconductor
  \author	  MSG
  \author	  IM, b06623
  \version	  2.0  
  \revision	  $Revision: 21 $
  \date  	  $Date: 2010-09-09 19:26:09 -0500 (Thu, 09 Sep 2010) $  

* History:
        - Updated Vertical Orientation. What it was really doing was 90 rendering
		  Now it does vertical text. Rotations should be handled after the text is written
	      by using 8bpp modes and fonts. May 25 2009
	    - 25 May 2010: Optimized usage of DMA, cleaned up code to make it more readable.

* MISRA VIOLATIONS:
	- [ MISRA 16.6 ] 
	- [ MISRA 16.9 ]

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
  
*/

#include	 "..\HIL\FontLibrary.h"
#include "..\HAL\DCU_drv.h"

Graphics_Object_t*	Font_GObject;
void*			Font_String;
Font_EncodingType	Font_Encoding;		
uint8_t 		Font_Length;
Font_FontType*		Font_Font;
Font_ModeType		Font_Mode;
Font_OrientationType	Font_Orient;
uint32_t		Font_AddressOffset;
Font_CallbackType	Font_Callback;
uint8_t			Font_4bppShift;
int16_t			Font_Xoffset;
Font_JustifyType 	Font_Justify;

/* May be used later when upgrading to other formats */
#if(0)
uint8_t			Font_Buffer[FONT_BUFFERSIZE];
#endif

uint8_t 			Font_Index;
uint16_t 			Font_Lines;
DMA_ChannelCfgType	FONT_DMAcfg;
uint8_t				Font_TaskState;
DMA_ErrorType	   	Font_DMAerror;

#define FONT_IDLE	(55u)
#define FONT_LOAD	(1u)
#define FONT_PRINT	(2u)
#define FONT_ENDING	(3u)

static void Font_StateMachine(uint8_t channel);

/**
* \brief	Font_Init - Initializes the font library control variables
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void Font_Init(void)
{
    Font_TaskState = FONT_IDLE;
    Font_Lines = 50u;
}

/**
* \brief	Font_SetLines - Configures the number of lines when using LSBFVS
* \author	IM, b06623
* \param	uint16_t lines, is the number of lines.
* \return	void
* \todo
*/
void Font_SetLines(uint16_t lines)
{
    Font_Lines = lines;
}

/**
* \brief	Font_PrintLeft - Prints a string on a graphic object, left aligned
* \brief	at the end of the execution the callback is called.
* \author	IM, b06623
* \param	uint8_t eDMAChannel, is the eDMA channel to use for transfers.
* \param	Graphics_Object_t*  g_object, is the graphic object to be used for printing.
* \param	uint8_t* string, is the pointer to the ASCII to be printed
* \param	uint8_t length, is the length of the string.
* \param	Font_FontType* font, is the font to be printed.
* \param	Font_CallbackType Callback, void *void Function to be called after the priting is done..
* \return	Font_ErrorType, FONT_ERROR_OK or FONT_ERROR_FAIL
* \todo
*/
Font_ErrorType Font_Print
(
	uint8_t eDMAChannel, Graphics_Object_t*  g_object,
	void* string, uint8_t length, Font_FontType* font, Font_EncodingType code,
	Font_ModeType mode, Font_OrientationType orientation, Font_JustifyType justify,
	Font_CallbackType callback, int16_t xoffset, uint16_t yoffset
)
{            
    DMA_ErrorType	error;
    uint16_t		roffset;
    uint32_t		divider;
    
	/* Check to be in correct Format */
    if( (font->format != FONT_8BPP) && (font->format != FONT_4BPP) )
    {
		return FONT_ERROR_FAIL;
    }
    /* Check to be FREE */
	if(Font_TaskState != FONT_IDLE)
	{
		return FONT_ERROR_FAIL;
	}	
	/* Reserve Channel */
	error = DMA_InitChannel(eDMAChannel);
	/* If not free, set everything back to normal */
	if( error != DMA_ERROR_OK )
	{
	    return FONT_ERROR_FAIL;
	}
	/* Everything is OK to do text rendering, go on */
    Font_4bppShift = 0;
    Font_Xoffset = xoffset;
    Font_Justify = justify;
    Font_GObject = g_object;
    Font_String = string;
    Font_Encoding = code;
    Font_Length = length;
    Font_Font = font;
    Font_Mode = mode;
    Font_Orient = orientation;
    /* VIOLATION TO [ MISRA 16.9 ] Rationale: This is the only way to configure a callback */ 
    Font_Callback = callback;
    Font_Index = 0u;
    Font_TaskState = FONT_PRINT;

    /* Byte offset */
    if(font->format == FONT_8BPP)
    {
		divider = 1u;
    }
    else
    {
		divider = 2u;
    }
    
    /* Calculate width */
    if(Font_Orient == FONT_HORIZONTAL)
    {
		roffset = Font_StringWidth(string, length, font, code, FONT_DMA);
    }
    else
    {
		roffset = 0u;
    }

    /* Calculate offsets due to alignment */
    if(justify == FONT_LEFT)
    { 
		Font_AddressOffset = ((uint32_t)xoffset/divider) + ((uint32_t)yoffset*((uint32_t)Font_GObject->width/divider));
    }
    else if(justify == FONT_CENTER)
    {
		Font_AddressOffset = (uint32_t)(xoffset + (((int16_t)Font_GObject->width - (int16_t)roffset)/2));
		Font_AddressOffset = (Font_AddressOffset/divider) + ((uint32_t)yoffset*((uint32_t)Font_GObject->width/divider)); 
    }
    else
    {
		Font_AddressOffset = (uint32_t)(xoffset + (int16_t)Font_GObject->width - (int16_t)roffset);
		Font_AddressOffset = (Font_AddressOffset/divider) +  ((uint32_t)yoffset*((uint32_t)Font_GObject->width/divider));
    }

    Font_StateMachine(eDMAChannel);

    return FONT_ERROR_OK;
}

/**
* \brief	Font_StringWidth - Returns the number of pixels length of the string 
* \author	IM, b06623
* \param	void* string, is the pointer to the ASCII or UNICODE to be printed
* \param	uint8_t length, is the length of the string.
* \param	Font_FontType* font, is the font to be printed.
* \param	Font_EncodingType code, select between ASCII or UNICODE
* \return	Font_EngineType eng, the engine used: DMA or CPU
* \return	uint16_t, returns the lenght in pixels
* \todo
*/
uint16_t Font_StringWidth(void* string, uint8_t length, Font_FontType* font, Font_EncodingType code, Font_EngineType eng)
{
    uint16_t i;
    uint16_t stringLen;
    uint16_t currentChar;

    stringLen = 0u;


    for(i = 0u; i < length; i++)
    {
    	/* use ascii or unicode */
    	if(code == FONT_ASCII)
		{
			currentChar = font->mapFunction(((uint8_t *)string)[i]);
		}
		else
		{
			currentChar = font->mapFunction(((uint16_t *)string)[i]);
		}
		/* check if kerned, and kerning data exist; add correction if yes */
		if((eng == FONT_KERNED) && (font->kerningData != NULL_PTR))
		{
		    stringLen += (uint16_t)font->kerningData[currentChar].xadvance;   
		}
		else
		{
		    stringLen += (uint16_t)font->GlyphSet[currentChar].pixelWidth;
		    if(eng == FONT_DMA)
		    {
				if(font->format == FONT_4BPP)
				{
				    if(font->GlyphSetShift != NULL_PTR)
				    {
						stringLen += 1u;
				    }
				    else
				    {		
						if(font->GlyphSet[currentChar].pixelWidth&0x0001u)
						{
					    	stringLen += 1u;
						}
				    }
				}
		    }
		}
    }   

    return stringLen;
}


static void Font_StateMachine(uint8_t channel)
{
    uint16_t currentChar;
    uint16_t divider;
    uint32_t minorLoopCount;
        
    /* state machine */
    if(Font_Index >= Font_Length)
    {
		Font_TaskState = FONT_ENDING;   
    }   
    if(Font_TaskState == FONT_PRINT)
    {
        /**********************************************************************************************************
    	 * The following lines set up the DMA values for the next Glyph transfer
    	 **********************************************************************************************************/ 
		divider = 1u;
		if(Font_Encoding == FONT_ASCII)
		{
			currentChar = Font_Font->mapFunction(((uint8_t *)Font_String)[Font_Index]);
		}
		else
		{
			currentChar = Font_Font->mapFunction(((uint16_t *)Font_String)[Font_Index]);
		}
		
		DMA_SourceOffset(channel) = 1;
		DMA_DestinationOffset(channel) = 1;
		DMA_SourceAddress(channel) = (uint32_t)(Font_Font->GlyphSet[currentChar].glyphAddress); 

		if(Font_Font->format == FONT_4BPP)
		{
		    divider = 2u;	    
		    if( (Font_4bppShift == 1) && (Font_Font->GlyphSetShift != NULL_PTR))
		    {
				DMA_SourceAddress(channel) = (uint32_t)(Font_Font->GlyphSetShift[currentChar].glyphAddress); 
				minorLoopCount = (uint32_t)(Font_Font->GlyphSetShift[currentChar].pixelWidth + 1u)/2u;
		    }
		    else
		    {
				minorLoopCount = (uint32_t)(Font_Font->GlyphSet[currentChar].pixelWidth + 1u)/2u;
		    }
		}
		else
		{
			/* case 8BPP */
			minorLoopCount = (uint32_t)(Font_Font->GlyphSet[currentChar].pixelWidth);
		}
		
		DMA_MinorLoopCount(channel) = minorLoopCount;		
	    DMA_MdestOffsetBit(channel) = 1;
		DMA_MsrcOffsetBit(channel) = 0;
		DMA_SetMajorLoopCount(channel,(uint16_t)(Font_Font->height));
		DMA_MloopOffsetValue(channel) = (int32_t)(Font_GObject->width/divider) - (int32_t)minorLoopCount;

		if( (Font_Font->GlyphSet[currentChar].pixelWidth&0x0001u) == 0u )
		{
		    /* If even width, toggle */
		    Font_4bppShift = (uint8_t)(Font_4bppShift^1u);
		}

		if(Font_Orient == FONT_HORIZONTAL)
		{
		    DMA_DestinationAddress(channel) = Font_GObject->address + Font_AddressOffset;  
		    Font_AddressOffset = Font_AddressOffset + minorLoopCount;
		}
		else
		{	    
		    /* Calculate offsets due to alignment */
		    if(Font_Justify == FONT_LEFT)
		    {
				DMA_DestinationAddress(channel) = Font_GObject->address + Font_AddressOffset; 
		    }
		    else if(Font_Justify == FONT_CENTER)
		    {
		    	DMA_DestinationAddress(channel) = Font_GObject->address + Font_AddressOffset - (minorLoopCount/2);
		    }
		    else
		    {	
		    	DMA_DestinationAddress(channel) = Font_GObject->address + Font_AddressOffset - minorLoopCount;
		    }
		    
		    Font_AddressOffset = Font_AddressOffset + ((uint32_t)Font_Font->height*((uint32_t)Font_GObject->width/divider));
		}
        /**********************************************************************************************************
    	 * This is the end of the DMA configuration!!
    	 **********************************************************************************************************/ 		

		/* VIOLATION TO [ MISRA 16.9 ] Rationale: This is the only way to configure a callback */   
		/* VIOLATION to [MISRA 16.2] Rationale: To obtain maximum effiency, this function must be recursive */  
		DMA_SetCallback(channel, Font_StateMachine);
		DMA_EnableRequest(channel, (uint8_t)(DMA_REQ_ALWAYS_START + channel));


       /**********************************************************************************************************
    	* Determine how and when to trigger start of DMA transfer depending on timing attributes
    	**********************************************************************************************************/ 
		
		if(Font_Mode == FONT_FREERUN)
		{
		    DMA_Start(channel);
		}
		else if(Font_Mode == FONT_VBLANK)
		{
		    /* check signals in DCU interrupt to go on or hold the execution */
		    //if( (Font_Index == 0u) || (DCU_GetTimingStatus( ) != DCU_VBLANK))
		    if(Font_Index == 0u)
		    {
				DMA_SetReady(channel);
		    }
		    else
		    {
				/* Still in the Vertical Blanking Interval */
				DMA_Start(channel);
		    }
		}
		else if(Font_Mode == FONT_LSBFVS)
		{
		    if(Font_Index == 0u)
		    {
				DMA_SetReady_LSBFVS(channel);
		    }
		    else
		    {
				DMA_Start(channel);
		    }		    		    
		}
		Font_Index++;
    }
    /* Terminate and release resources */
    else if(Font_TaskState == FONT_ENDING)
    {
		DMA_UnInitChannel(channel);	
		Font_TaskState = FONT_IDLE;
		/* VIOLATION TO [ MISRA 11.1 ] Rationale: NULL_PTR will be the standard null type for both objects and functions */   
		if(Font_Callback != NULL_PTR)
		{
	    	Font_Callback();
		}	
    } 
}


/**********************************************************************************************************
* Starting from here, everything belongs to CPU text rendering functions!!
**********************************************************************************************************/ 


Graphics_Object_t*	Font_GObject1;
void*			Font_String1;
Font_EncodingType	Font_Encoding1;	
uint8_t 		Font_Length1;
Font_FontType*		Font_Font1;
Font_OrientationType	Font_Orient1;
uint32_t		Font_AddressOffset1;
uint8_t			Font_4bppShift1;
int16_t			Font_Xoffset1;
Font_JustifyType 	Font_Justify1;
uint8_t 		Font_Index1;

void Font_PrintCPU
(
	Graphics_Object_t*  g_object, void* string, uint8_t length,
	Font_FontType* font, Font_EncodingType code, Font_ModeType mode, Font_OrientationType orientation,
	Font_JustifyType justify, int16_t xoffset, uint16_t yoffset
)
{    
    uint16_t roffset;
    Font_EngineType engType;
    
    Font_4bppShift1 = 0;
    Font_Xoffset1 = xoffset;
    Font_Justify1 = justify;
    Font_GObject1 = g_object;
    Font_String1 = string;
    Font_Encoding1 = code;
    Font_Length1 = length;
    Font_Font1 = font;
    Font_Orient1 = orientation;
    Font_Index1 = 0u;
    
    /* Calculate width of string */    
    if(Font_Orient1 == FONT_HORIZONTAL)
    {
		if(font->kerningData != NULL_PTR)
		{
	    	engType = FONT_KERNED;
		}
		else
		{
	    	engType = FONT_CPU;
		}
	
		roffset = Font_StringWidth(string, length, font, code, engType);
    }
    else
    {
		roffset = 0u;
    }

    /* Calculate offsets due to alignment */
    if(justify == FONT_LEFT)
    { 
		Font_AddressOffset1 = (uint32_t)xoffset + ((uint32_t)yoffset*(uint32_t)Font_GObject1->width);
    }
    else if(justify == FONT_CENTER)
    {
		Font_AddressOffset1 = (uint32_t)(xoffset + (((int16_t)Font_GObject1->width - (int16_t)roffset)/2));
		Font_AddressOffset1 = Font_AddressOffset1 + ((uint32_t)yoffset*(uint32_t)Font_GObject1->width); 
    }
    else
    {
		Font_AddressOffset1 = (uint32_t)(xoffset + (int16_t)Font_GObject1->width - (int16_t)roffset);
		Font_AddressOffset1 = Font_AddressOffset1 +  ((uint32_t)yoffset*(uint32_t)Font_GObject1->width);
    }

    if(mode == FONT_FREERUN)
    {
		Font_CPU_PrintMachine(length);
    }

}

/**********************************************************************************************************
* These macros allows an standar way to set or clear pixels on different pixel sizes
**********************************************************************************************************/ 

#define FONT_PIXEL8BPP(address, width, x, y)		(* (uint8_t *)((((int32_t)(address)) + (x)) + ((y)*(width))))
#define FONT_BYTEPIXEL4BPP(address, width, x, y)	(* (uint8_t *)((((int32_t)(address)) + ((x)/2)) + ((y)*((width+1)>>1))))
#define FONT_BYTEPIXEL2BPP(address, width, x, y)	(* (uint8_t *)((((int32_t)(address)) + ((x)/4)) + ((y)*((width+3)>>2))))
#define FONT_BYTEPIXEL1BPP(address, width, x, y)	(* (uint8_t *)((((int32_t)(address)) + ((x)/8)) + ((y)*((width+7)>>3))))
#define FONT_CUT1BYTEPIXEL1BPP(address, width, x, y)	(* (uint8_t *)(((((int32_t)(address)) + ((x)/8)) + ((y)*(width>>3)))^2))


/**********************************************************************************************************
* CPU Print Machine for printing text. You tell how many chars to print
**********************************************************************************************************/ 

void Font_CPU_PrintMachine(uint8_t nChars)
{
    uint16_t w;
    uint16_t h;
    uint16_t width;
    uint16_t width2;
    uint16_t height;
    uint16_t currentChar;
    uint32_t tmpPixel;
    uint32_t tmpPixel2;
    
    uint8_t *ptrD;
    uint8_t *ptrS;
    uint8_t bitShift;
    uint8_t bitShift2;

    
    /* Verify if task is done */
    if(Font_Index1 < Font_Length1)
    {
		while(nChars != 0)
		{
		    /* Two branches, kerning and no-kerning */
		    /**********************************************************************************************************
			* CPU Print non KERNED text
			**********************************************************************************************************/ 
		    if(Font_Font1->kerningData != NULL_PTR)
		    {
			    if(Font_Encoding1 == FONT_ASCII)
				{
					currentChar = Font_Font1->mapFunction(((uint8_t *)Font_String1)[Font_Index1]);
				}
				else
				{
					currentChar = Font_Font1->mapFunction(((uint16_t *)Font_String1)[Font_Index1]);
				}
				width = Font_Font1->kerningData[currentChar].width;
				width2 = Font_Font1->GlyphSet[currentChar].pixelWidth;
				height = Font_Font1->kerningData[currentChar].height;
				
				if(Font_Orient1 == FONT_VERTICAL)
				{
				    if(Font_Justify1 == FONT_CENTER)
				    {
						Font_AddressOffset1 = Font_AddressOffset1 - Font_Font1->kerningData[currentChar].xadvance/2;
				    }
				    else if(Font_Justify1 == FONT_RIGHT)
				    {
						Font_AddressOffset1 = Font_AddressOffset1 - Font_Font1->kerningData[currentChar].xadvance;
				    }			    
				}

				if( ((int16_t)Font_AddressOffset1 + Font_Font1->kerningData[currentChar].xoffset) < 0)
				{
				    Font_AddressOffset1 = (uint16_t)(-1*Font_Font1->kerningData[currentChar].xoffset);    
				}
				switch(Font_Font1->format)
				{
				    /* CASE 8BPP text */
				    case FONT_8BPP:
						for(h = 0u; h <height; h++)
						{
						    for(w = 0u; w < width; w++)
						    {				
								tmpPixel = FONT_PIXEL8BPP((uint32_t)Font_Font1->GlyphSet[currentChar].glyphAddress, width2, w + Font_Font1->kerningData[currentChar].x, h + Font_Font1->kerningData[currentChar].y);
								FONT_PIXEL8BPP((uint32_t)Font_GObject1->address + Font_AddressOffset1, Font_GObject1->width, (int16_t)w + Font_Font1->kerningData[currentChar].xoffset, h + Font_Font1->kerningData[currentChar].yoffset) |= (uint8_t)tmpPixel;
						    }
						}
					break;
					/* CASE 4BPP text */
				    case FONT_4BPP:
						for(h = 0u; h <height; h++)
						{
						    for(w = 0u; w < width; w++)
						    {
								tmpPixel = FONT_BYTEPIXEL4BPP((uint32_t)Font_Font1->GlyphSet[currentChar].glyphAddress, width2, w + Font_Font1->kerningData[currentChar].x, h + Font_Font1->kerningData[currentChar].y);
								if((w + Font_Font1->kerningData[currentChar].x)&1u)
								{
								    tmpPixel = tmpPixel>>4u;
								}
								else
								{
								    tmpPixel = tmpPixel&0x0Fu;   
								}
								tmpPixel2 = FONT_BYTEPIXEL4BPP((uint32_t)Font_GObject1->address, Font_GObject1->width, (int16_t)w + Font_Font1->kerningData[currentChar].xoffset + (int16_t)Font_AddressOffset1,h + Font_Font1->kerningData[currentChar].yoffset);
								if( ((uint16_t)((int16_t)w + Font_Font1->kerningData[currentChar].xoffset + (int16_t)Font_AddressOffset1)) & 1u)
								{
								    tmpPixel2 = (tmpPixel<<4)|(0x0Fu&tmpPixel2); 
								}
								else
								{
								    tmpPixel2 = (tmpPixel)|(0xF0u&tmpPixel2);
								}
								FONT_BYTEPIXEL4BPP((uint32_t)Font_GObject1->address, Font_GObject1->width, (int16_t)w + Font_Font1->kerningData[currentChar].xoffset + (int16_t)Font_AddressOffset1, h + Font_Font1->kerningData[currentChar].yoffset) = (uint8_t)tmpPixel2;
						    }
						}
					break;
					/* CASE 2BPP text */	
				    case FONT_2BPP:
				    	width = Font_Font1->GlyphSet[currentChar].pixelWidth;
				    	if(Font_Font1->kerningData[currentChar].xoffset == -1)
				    	{
				    		if(Font_AddressOffset1 != 0)
				    		{
				    			Font_AddressOffset1--;
				    		}
				    	}		    		
				    	ptrD = (uint8_t *)(Font_GObject1->address + (Font_AddressOffset1>>2) + (Font_Font1->kerningData[currentChar].y*(Font_GObject1->width>>2)));
				    	bitShift = (uint8_t)((Font_AddressOffset1&3)<<1);		    	
				    	bitShift2 = (uint8_t)(8 - bitShift);
				    	width2 = (uint16_t)(1 + (width>>2));
				    	if((width&3) == 0)
				    	{
				    		width2--;
				    	}
				    	
				    	ptrS = (uint8_t *)(((uint32_t)Font_Font1->GlyphSet[currentChar].glyphAddress) 
							   + ((uint32_t)(Font_Font1->kerningData[currentChar].y*width2)));
						height = (uint16_t)(Font_Font1->height - Font_Font1->kerningData[currentChar].y);							    		
						for(h = 0; h <height; h++)
						{
						    for(w = 0u; w < width2; w++)
						    {
								*ptrD |= (*ptrS<<bitShift);
								ptrD = ptrD + 1;
								if(bitShift != 0)
								{
									*ptrD = (uint8_t)(*ptrS>>bitShift2);
								} 
								ptrS++;
						    }
						    ptrD += (Font_GObject1->width>>2) - width2;
						}			
					break;	
					/* CASE 1BPP text */
				    case FONT_1BPP:
				    	width = Font_Font1->GlyphSet[currentChar].pixelWidth;
				    	if(Font_Font1->kerningData[currentChar].xoffset == -1)
				    	{
				    		if(Font_AddressOffset1 != 0)
				    		{
				    			Font_AddressOffset1--;
				    		}
				    	}
				    		
				    	ptrD = (uint8_t *)(Font_GObject1->address + (Font_AddressOffset1>>3) + (Font_Font1->kerningData[currentChar].y*(Font_GObject1->width>>3)));
				    	bitShift = (uint8_t)(Font_AddressOffset1&7);		    	
				    	bitShift2 = (uint8_t)(8 - bitShift);
				    	width2 = (uint16_t)(1 + (width>>3));
				    	if((width&7) == 0)
				    	{
				    		width2--;
				    	}
				    	
				    	ptrS = (uint8_t *)(((uint32_t)Font_Font1->GlyphSet[currentChar].glyphAddress) 
							   + ((uint32_t)(Font_Font1->kerningData[currentChar].y*width2)));
						height = (uint16_t)(Font_Font1->height - Font_Font1->kerningData[currentChar].y);				    		
						for(h = 0; h <height; h++)
						{
						    for(w = 0u; w < width2; w++)
						    {
								*ptrD |= (*ptrS<<bitShift);
								ptrD = ptrD + 1;
								if(bitShift != 0)
								{
									*ptrD = (uint8_t)(*ptrS>>bitShift2);
								} 
								ptrS++;
						    }
						    ptrD += (Font_GObject1->width>>3) - width2;
						}			
					break;												
				    default:
					break;
				}
				if(Font_Orient1 == FONT_VERTICAL)
				{
				    if(Font_Justify1 == FONT_CENTER)
				    {
						Font_AddressOffset1 = Font_AddressOffset1 + Font_Font1->kerningData[currentChar].xadvance/2;
				    }
				    else if(Font_Justify1 == FONT_RIGHT)
				    {
						Font_AddressOffset1 = Font_AddressOffset1 + Font_Font1->kerningData[currentChar].xadvance;
				    }
				    Font_AddressOffset1 = Font_AddressOffset1 + (Font_Font1->height*Font_GObject1->width);			    
				}
				else
				{
				    Font_AddressOffset1 = Font_AddressOffset1 + Font_Font1->kerningData[currentChar].xadvance;
				}

			    }
		    /**********************************************************************************************************
			* CPU Print non KERNED text
			**********************************************************************************************************/ 		    
		    else
		    {
				if(Font_Encoding1 == FONT_ASCII)
				{
					currentChar = Font_Font1->mapFunction(((uint8_t *)Font_String1)[Font_Index1]);
				}
				else
				{
					currentChar = Font_Font1->mapFunction(((uint16_t *)Font_String1)[Font_Index1]);
				}
				width = Font_Font1->GlyphSet[currentChar].pixelWidth;
				if(Font_Orient1 == FONT_VERTICAL)
				{
				    if(Font_Justify1 == FONT_CENTER)
				    {
						Font_AddressOffset1 = Font_AddressOffset1 - width/2;
				    }
				    else if(Font_Justify1 == FONT_RIGHT)
				    {
						Font_AddressOffset1 = Font_AddressOffset1 - width;
				    }			    
				}
				switch(Font_Font1->format)
				{
					/* Case 8BPP font Kerned */
				    case FONT_8BPP:
						for(h = 0; h <Font_Font1->height; h++)
						{
						    for(w = 0u; w < width; w++)
						    {
								tmpPixel = FONT_PIXEL8BPP((uint32_t)Font_Font1->GlyphSet[currentChar].glyphAddress, width, w, h);
								FONT_PIXEL8BPP((uint32_t)Font_GObject1->address + Font_AddressOffset1, Font_GObject1->width, w, h) = (uint8_t)tmpPixel;
						    }
						}
					break;
					/* Case 4BPP font Kerned */
				    case FONT_4BPP:
						for(h = 0; h <Font_Font1->height; h++)
						{
						    for(w = 0u; w < width; w++)
						    {
								tmpPixel = FONT_BYTEPIXEL4BPP((uint32_t)Font_Font1->GlyphSet[currentChar].glyphAddress, width, w, h);
								if(w&1u)
								{
								    tmpPixel = tmpPixel>>4u;
								}
								else
								{
								    tmpPixel = tmpPixel&0x0Fu;   
								}
								tmpPixel2 = FONT_BYTEPIXEL4BPP((uint32_t)Font_GObject1->address, Font_GObject1->width, w + Font_AddressOffset1, h);
								if((w + Font_AddressOffset1)&1u)
								{
								    tmpPixel2 = (tmpPixel<<4)|(0x0Fu&tmpPixel2); 
								}
								else
								{
								    tmpPixel2 = (tmpPixel)|(0xF0u&tmpPixel2);
								}
								FONT_BYTEPIXEL4BPP((uint32_t)Font_GObject1->address, Font_GObject1->width, w + Font_AddressOffset1, h) = (uint8_t)tmpPixel2;
						    }
						}
					break;
					/* Case 2BPP font Kerned */
				    case FONT_2BPP:
						ptrS = (uint8_t *)((uint32_t)Font_Font1->GlyphSet[currentChar].glyphAddress);
					    ptrD = (uint8_t *)(Font_GObject1->address + (Font_AddressOffset1>>2));
					    bitShift = (uint8_t)((Font_AddressOffset1&3)<<1);
					    bitShift2 = (uint8_t)(8 - bitShift);
					    width2 = (uint16_t)(1 + (width>>2));		    
					    if((width&3) == 0)
					    {
					    	width2--;
					    }		    		
						for(h = 0; h <Font_Font1->height; h++)
						{
						    for(w = 0u; w < width2; w++)
						    {
								*ptrD |= (*ptrS<<bitShift);
								ptrD = ptrD + 1;
								if(bitShift != 0)
								{
									*ptrD = (uint8_t)(*ptrS>>bitShift2);
								} 
								ptrS++;
						    }
						    ptrD += (Font_GObject1->width>>2) - width2;
						}
					break;	
					/* Case 1BPP font Kerned */	
					case FONT_1BPP:
						ptrS = (uint8_t *)((uint32_t)Font_Font1->GlyphSet[currentChar].glyphAddress);
					    ptrD = (uint8_t *)(Font_GObject1->address + (Font_AddressOffset1>>3));
					    bitShift = (uint8_t)(Font_AddressOffset1&7);
					    bitShift2 = (uint8_t)(8 - bitShift);
					    width2 = (uint16_t)(1 + (width>>3));
					    if((width%8) == 0)
					    {
					    	width2--;
					    }		    		
						for(h = 0; h <Font_Font1->height; h++)
						{
						    for(w = 0u; w < width2; w++)
						    {
								*ptrD |= (*ptrS<<bitShift);
								ptrD = ptrD + 1;
								if(bitShift != 0)
								{
									*ptrD = (uint8_t)(*ptrS>>bitShift2);
								} 
								ptrS++;
						    }
						    ptrD += (Font_GObject1->width>>3) - width2;
						}
					break;							
				    default:
					break;
				}
				
				if(Font_Orient1 == FONT_VERTICAL)
				{
				    if(Font_Justify1 == FONT_CENTER)
				    {
						Font_AddressOffset1 = Font_AddressOffset1 + width/2;
				    }
				    else if(Font_Justify1 == FONT_RIGHT)
				    {
						Font_AddressOffset1 = Font_AddressOffset1 + width;
				    }
				    Font_AddressOffset1 = Font_AddressOffset1 + (Font_Font1->height*Font_GObject1->width);			    
				}
				else
				{
				    Font_AddressOffset1 = Font_AddressOffset1 + width;
				}


			    }
		    Font_Index1++;
		    nChars--;
		}	 
    }
}
